﻿using System;
using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.EventSystems;
using UnityEngine.SceneManagement;

public class PauseMenu : MonoBehaviour
{
    public GameObject PauseMenuUI;

    private GameObject _BackgroundPanel;

    public static bool CanPause { get; set; } = true;

    private bool _IsAnimating = false;

    
    void Start()
    {
        //Subscribe to events
        InputManager.Instance.InputStateChanged += OnInputStateChanged;
        _BackgroundPanel = PauseMenuUI.FindChild("BackgroundPanel");
        _BackgroundPanel.GetComponent<PhaseAnimation>().BounceInCompleted += PauseMenu_BounceInCompleted;
        _BackgroundPanel.GetComponent<PhaseAnimation>().BounceOutCompleted += PauseMenu_BounceOutCompleted;
    }

    private void PauseMenu_BounceInCompleted(object sender, EventArgs e)
    {
        if (InputManager.Instance.InputState == Constants.InputState.Controller)
        {
            //Just changed to controller, focus on the resume button
            EventSystem.current.SetSelectedGameObject(null);
            EventSystem.current.SetSelectedGameObject(_BackgroundPanel.FindChild("ResumeButton"));
        }

        //Bouncing is done so enable the UI and we're no longer animating
        PauseMenuUI.Enable();
        _IsAnimating = false;
    }

    private void PauseMenu_BounceOutCompleted(object sender, EventArgs e)
    {
        //Bouncing is done so hide the pause UI, unpause, reactivate the players and the game UI, and we're no longer animating
        PauseMenuUI.SetActive(false);
        Time.timeScale = 1.0f;
        GameController.Instance.HiderPlayerObject.GetComponent<BasicMovement>().CanMove = true;
        GameController.Instance.SeekerPlayerObject.GetComponent<BasicMovement>().CanMove = true;
        GameController.Instance.Timer.GetComponent<Animator>().enabled = true;
        GameController.Instance.Timer.Show();

        if (GameController.Instance.CurrentGamePhase.GetType() == typeof(SeekingPhase))
        {
            GameController.Instance.HintUI.SetActive(true);
        }

        GameManager.Instance.GameState = Constants.GameState.Running;
        _IsAnimating = false;
    }

    private void OnInputStateChanged(object sender, InputManager.InputStateChangedEventArgs e)
    {
        if (e.NewState == Constants.InputState.Controller && gameObject.activeSelf)
        {
            //We've just changed to a controller so focus on resume
            EventSystem.current.SetSelectedGameObject(null);
            EventSystem.current.SetSelectedGameObject(_BackgroundPanel.FindChild("ResumeButton"));
            e.Handled = true;
        }
    }

    private void OnDestroy()
    {
        //Unsubscribe from events
        InputManager.Instance.InputStateChanged -= OnInputStateChanged;
        _BackgroundPanel.GetComponent<PhaseAnimation>().BounceInCompleted -= PauseMenu_BounceInCompleted;
        _BackgroundPanel.GetComponent<PhaseAnimation>().BounceOutCompleted -= PauseMenu_BounceOutCompleted;
    }
    
    void Update()
    {
        if (!_IsAnimating)
        {
            //We're not animating so check for input
            if (Input.GetKeyDown(KeyCode.Escape) || Input.GetKeyDown("joystick button 7"))
            {
                //The pause/unpause button was pressed, switch the state as appropriate
                if (GameManager.Instance.GameState == Constants.GameState.Paused)
                {
                    Resume();
                }

                else
                {
                    Pause();
                }
            }
        }
    }

    private void Pause()
    {
        if (CanPause)
        {
            //We're allowed to pause, so enable the pause UI, disable the characters and game UI and begin animating the panel in
            PauseMenuUI.SetActive(true);
            _BackgroundPanel.Disable();
            Time.timeScale = 0.0f;
            _BackgroundPanel.gameObject.transform.localScale = new Vector3(0.0f, 0.0f, 1.0f);
            _IsAnimating = true;
            _BackgroundPanel.GetComponent<Animator>().Play("BounceIn", -1, 0.0f);
            GameController.Instance.HiderPlayerObject.GetComponent<BasicMovement>().CanMove = false;
            GameController.Instance.SeekerPlayerObject.GetComponent<BasicMovement>().CanMove = false;
            GameController.Instance.Timer.GetComponent<Animator>().enabled = false;
            GameController.Instance.Timer.Hide();
            GameController.Instance.HintUI.SetActive(false);
            EventSystem.current.SetSelectedGameObject(null);
            GameManager.Instance.GameState = Constants.GameState.Paused;
        }
    }


    public void Resume()
    {
        //We're resuming so disable the UI, we're animating so ignore user input now and begin animating the panel out
        PauseMenuUI.Disable();
        _IsAnimating = true;
        _BackgroundPanel.GetComponent<Animator>().Play("BounceOut", -1, 0.0f);
    }

    public void QuitToMainMenu()
    {
        //Reset the music audio levels, hide the pause menu UI, reset the time scale and load back into the menu
        AudioManager.Instance.SetMusicVolume(Convert.ToSingle(SettingsManager.Instance.Settings[Constants.Settings.MusicVolume]));  //We need to do this so if we're paused, the audio is reset back to its proper value
        PauseMenuUI.SetActive(false);
        Time.timeScale = 1.0f;
        GameManager.Instance.GameState = Constants.GameState.Running;
        SceneManager.LoadScene("Menu");
    }
}
